# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.


from odoo import models, fields, api, _
from odoo.exceptions import ValidationError
from datetime import datetime
from odoo.tools import DEFAULT_SERVER_DATE_FORMAT, float_compare

class AccountInvoice(models.Model):
    _inherit = 'account.invoice'

    new_fiscal_position_id = fields.Many2one('account.fiscal.position', string='Temporary fiscal position', readonly=True, states={'draft': [('readonly', False)]})

    @api.onchange('partner_id', 'company_id', 'date_invoice')
    def _onchange_partner_id(self):
        result = super(AccountInvoice, self)._onchange_partner_id()
        self.new_fiscal_position_id = False
        if self.partner_id and self.partner_id.new_property_account_position_id:
            date_ref = datetime.now().date().strftime('%Y-%m-%d')
            if self.date_invoice:
                date_ref = str(datetime.strptime(self.date_invoice, DEFAULT_SERVER_DATE_FORMAT))
            from_date = self.company_id.fiscal_position_date_start
            to_date = self.company_id.fiscal_position_date_stop
            #check date limit
            if (from_date and to_date and from_date <= date_ref and to_date >= date_ref) or (not from_date and to_date and to_date >= date_ref) or (from_date and not to_date and from_date <= date_ref) or (not from_date and not to_date):
                self.new_fiscal_position_id = self.partner_id.new_property_account_position_id.id
            else:
                self.new_fiscal_position_id = False
        return result

    @api.model
    def apply_fiscal_position(self):
        fiscal_pool = self.env['account.fiscal.position']
        context = dict(self._context)
        for record in self.search([('id', 'in', context.get('active_ids'))]):
            if record.state != 'draft':
                raise ValidationError(_("You can only apply temporary fiscal position on draft records."))
            if record.partner_id.new_property_account_position_id:
                record.new_fiscal_position_id = record.partner_id.new_property_account_position_id.id
                for line in record.invoice_line_ids:
                    #Apply temporary tax
                    taxes = line.invoice_line_tax_ids
                    line.invoice_line_tax_ids = record.partner_id.new_property_account_position_id.map_tax(taxes, line.product_id, line.invoice_id.partner_id) if record.partner_id.new_property_account_position_id else taxes
                    #Apply temporary account
                    account_id = line.get_invoice_line_account(record.type, line.product_id, record.new_fiscal_position_id, record.company_id)
                    if account_id:
                        line.account_id = account_id.id
            record.compute_taxes()
        return True

    @api.model
    def apply_old_tax(self):
        fiscal_pool = self.env['account.fiscal.position']
        context = dict(self._context)
        for record in self.search([('id', 'in', context.get('active_ids'))]):
            if record.state != 'draft':
                raise ValidationError(_("You can only apply old tax on draft records."))
            new_fiscal_position = record.new_fiscal_position_id
            if new_fiscal_position:
                for line in record.invoice_line_ids:
                    taxes = []
                    for tax in line.invoice_line_tax_ids:
                        taxes.append(tax.id)
                        for fo_tax in new_fiscal_position.tax_ids:
                            if fo_tax.tax_dest_id and tax.id == fo_tax.tax_dest_id.id:
                                if tax.id in taxes:
                                    taxes.remove(tax.id)
                                taxes.append(fo_tax.tax_src_id.id)    
                    line.invoice_line_tax_ids = [(6, 0, list(set(taxes)))]
                    account_id = line.account_id.id
                    for fo_account in new_fiscal_position.account_ids:
                        if line.account_id.id == fo_account.account_dest_id.id:
                            account_id = fo_account.account_src_id.id
                    line.account_id = account_id
            record.new_fiscal_position_id = False
            record.compute_taxes()
        return True

    @api.onchange('purchase_id')
    def purchase_order_change(self):
        result = super(AccountInvoice, self).purchase_order_change()
        if self.partner_id and self.partner_id.new_property_account_position_id:
            self.new_fiscal_position_id = self.partner_id.new_property_account_position_id.id
        return result

    def _prepare_invoice_line_from_po_line(self, line):
        result = super(AccountInvoice, self)._prepare_invoice_line_from_po_line(line=line)
        if self.purchase_id.new_fiscal_position_id:
            if result.get('account_id'):
                account_id = self.env['account.account'].sudo().browse(result.get('account_id'))
                account = self.purchase_id.new_fiscal_position_id.map_account(account_id)
                if account:
                    result['account_id'] = account.id
            if result.get('invoice_line_tax_ids'):
                taxes = self.env['account.tax'].sudo().browse(result.get('invoice_line_tax_ids'))
                product_id = self.env['product.product'].sudo().browse(result.get('product_id'))
                result['invoice_line_tax_ids'] = self.purchase_id.new_fiscal_position_id.map_tax(taxes, product_id, self.purchase_id.partner_id)
        return result

    @api.multi
    @api.returns('self')
    def refund(self, date_invoice=None, date=None, description=None, journal_id=None):
        result = super(AccountInvoice, self).refund(date_invoice=date_invoice, date=date, description=description, journal_id=journal_id)
        for invoice in self:
            result.write({'new_fiscal_position_id': invoice.new_fiscal_position_id and invoice.new_fiscal_position_id.id or False})
        return result

class AccountInvoiceLine(models.Model):
    _inherit = 'account.invoice.line'

    @api.model
    def _default_account(self):
        account_id = super(AccountInvoiceLine, self)._default_account()
        if self._context.get('new_fiscal_position_id') and account_id:
            fpos_id = self.env['account.fiscal.position'].sudo().browse(self._context.get('new_fiscal_position_id'))
            account_id = self.env['account.account'].sudo().browse(account_id)
            fpos_account = fpos_id.map_account(account_id)
            if fpos_account:
                return fpos_account.id
        return account_id

    account_id = fields.Many2one('account.account', string='Account', domain=[('deprecated', '=', False)],
        default=_default_account,
        help="The income or expense account related to the selected product.")

    @api.onchange('account_id')
    def _onchange_account_id(self):
        if not self.account_id:
            return
        if not self.product_id:
            fpos = self.invoice_id.new_fiscal_position_id or self.invoice_id.fiscal_position_id
            self.invoice_line_tax_ids = fpos.map_tax(self.account_id.tax_ids, partner=self.partner_id).ids
        elif not self.price_unit:
            self._set_taxes()

    def _set_taxes(self):
        result = super(AccountInvoiceLine, self)._set_taxes()
        for line in self:
            if line.invoice_id.new_fiscal_position_id:
                taxes = line.invoice_line_tax_ids
                line.invoice_line_tax_ids = line.invoice_id.new_fiscal_position_id.map_tax(taxes, line.product_id, line.invoice_id.partner_id)
        return result

    @api.v8
    def get_invoice_line_account(self, type, product, fpos, company):
        if self.invoice_id.new_fiscal_position_id:
            accounts = product.product_tmpl_id.get_product_accounts(self.invoice_id.new_fiscal_position_id)
            if type in ('out_invoice', 'out_refund'):
                return accounts['income']
            return accounts['expense']
        else:
            return super(AccountInvoiceLine, self).get_invoice_line_account(type=type, product=product, fpos=fpos, company=company)

class AccountFiscalPosition(models.Model):
    _inherit = 'account.fiscal.position'
    _description = 'Fiscal Position'
    _order = 'sequence'

    related_fiscal_position_id = fields.Many2one('account.fiscal.position', string='Temporary fiscal position')
    default = fields.Boolean(string='Default temp. fiscal position')
